"""
    ace_reports.py
        Script to do the following:
            1. Download reports from am ACE controller via FTP.
            2. Archive the reports locally with date-stamped filenames.
            3. Process the transaction summary log report into an HTML-ified t-log and a security summary report
            containing only transactions of security interest (containing voids, refunds, discounts, IS cancels, and
            suspends).
            4. Upload reports marked for upload to a remote SFTP server.

jA 250426
"""
import ftplib
import io
import logging
from logging.handlers import TimedRotatingFileHandler
import re
import time
from dataclasses import dataclass
from datetime import date
from ftplib import FTP
from pathlib import Path
from typing import List, Tuple, Callable
from typing import Optional, Pattern
from typing import TypeVar, Dict, NamedTuple

from paramiko.sftp_client import SFTPClient
from pyhocon import ConfigFactory

# Import shared utilities
try:
    # Try relative import first (when used as a package)
    from .commserv_utils import (
        Server, StoreSettings, setup_logging, get_logger, with_retry,
        with_ftp_connection, with_sftp_connection, write_str_to_path, get_store_settings
    )
except ImportError:
    # Fall back to absolute import (when run as a script)
    from commserv_utils import (
        Server, StoreSettings, setup_logging, get_logger, with_retry,
        with_ftp_connection, with_sftp_connection, write_str_to_path, get_store_settings
    )

T = TypeVar('T')


@dataclass
class FileSpec:
    """Specifies a report file to be processed.

    Attributes:
        name (str): Display name of the report
        path (str): Path to the report file on remote system
        id (str): Unique identifier for the report. Used as prefix for archived filename
        upload (bool): Whether to upload file to remote host, defaults to False
        required (bool): Whether the report file must exist, defaults to True
    """
    name: str
    path: str
    id: str
    upload: bool = False
    required: bool = True


# Constants
LOGGER_NAME = "ace-reports"
NETWORK_TIMEOUT = 30


@dataclass
class Settings(StoreSettings):
    """Settings for the ace_reports script.

    Inherits all shared settings from StoreSettings and adds application-specific settings.

    Attributes:
        log_base_filename (str): Base filename for logs
        reports_server (Server): Reports server configuration
        local_reports_root (str): Path to the local reports root
        tlog_id (str): ID of the transaction log report
        vrdis_id (str): ID of the VRDIS report
        vrdis_name (str): Name of the VRDIS report
        vrdis_inclusion_patterns (List[str]): List of VRDIS transaction inclusion patterns
        reports (List[FileSpec]): List of reports to process
    """
    log_base_filename: str
    reports_server: Server
    local_reports_root: str
    tlog_id: str
    vrdis_id: str
    vrdis_name: str
    vrdis_inclusion_patterns: List[str]
    reports: List[FileSpec]


class Transaction(NamedTuple):
    """Represents an indexed transaction."""
    index: int
    detail: str

    def detail_html(self) -> str:
        """Return transaction detail as HTML with index and styling.

        Returns:
            str: HTML-formatted transaction block
        """
        return (
            f'<a name="{self.index}"></a>'
            f'<PRE>{self.index} {"--" * 37}\n'
            f'{self.detail}'
            f'</PRE>\n\n'
        )


class UploadQueueItem(NamedTuple):
    """Represents an item in the upload queue.

    Attributes:
        content (str): Upload content
        filename (str): Upload filename
    """
    content: str
    filename: str


def load_settings() -> Settings:
    """Load configuration from HOCON files.

    Loads shared settings from store.conf and application-specific settings from ace_reports.conf.

    Returns:
        Settings: Combined configuration

    Raises:
        FileNotFoundError: If any config file doesn't exist
        ConfigException: If any config file has invalid format
    """
    # Get shared store settings
    store_settings = get_store_settings()
    
    # Load app-specific settings
    config_path = Path(__file__).parent.parent / "etc" / "ace_reports.conf"
    config = ConfigFactory.parse_file(str(config_path))
    
    # Create combined settings
    return Settings(
        # Shared settings from store.conf
        store_number=store_settings.store_number,
        cc=store_settings.cc,
        network_timeout=store_settings.network_timeout,
        retry_interval=store_settings.retry_interval,
        max_attempts=store_settings.max_attempts,
        log_directory=store_settings.log_directory,
        log_count=store_settings.log_count,
        
        # App-specific settings from ace_reports.conf
        log_base_filename=config.get_string("log_base_filename"),
        reports_server=Server(
            protocol=config.get_string("reports_server.protocol"),
            host=config.get_string("reports_server.host"),
            username=config.get_string("reports_server.username"),
            password=config.get_string("reports_server.password"),
            port=config.get_int("reports_server.port"),
            path=config.get_string("reports_server.path")
        ),
        local_reports_root=config.get_string("local_reports_root"),
        tlog_id=config.get_string("tlog_id"),
        vrdis_id=config.get_string("vrdis.id"),
        vrdis_name=config.get_string("vrdis.name"),
        vrdis_inclusion_patterns=config.get_list("vrdis.inclusion_patterns"),
        reports=[
            FileSpec(
                name=spec.get_string("name"),
                path=spec.get_string("path"),
                id=spec.get_string("id"),
                upload=spec.get_bool("upload", False),
                required=spec.get_bool("required", True)
            ) for spec in config.get_list("reports")
        ]
    )


TLOG_TX_SEPARATOR = '=' * 80

# report date pattern
REPORT_DATE_PATTERN = re.compile(r"REPORTED AT:*\s+(\d{2})/(\d{2})/(\d{2})", re.IGNORECASE)

# patterns to filter out unwanted content from t-logs
TLOG_FILTER_PATTERNS = [
    re.compile(pattern) for pattern in [
        r"^\s*Page \d+",  # Page numbers
        r"^\s*Auto Report: .*",  # Auto Report header
        r"^\s*TRANSACTION SUMMARY LOG REPORT.*",  # Report title
        r"^\s*PREVIOUS PERIOD.*",  # Previous period line
        r"^\s*Reported at:.*",  # Report timestamp
        r"^-+$",  # Separator lines
        r"^\s*$"  # Empty lines
    ]
]

TLOG_DATE_PATTERN = re.compile(r"\d{2}/\d{2}/\d{2}")


def get_application_logger():
    """Get the application logger."""
    return get_logger(LOGGER_NAME)


# At module level
_settings = None


def get_settings():
    """Get application settings, loading them if necessary."""
    global _settings
    if _settings is None:
        _settings = load_settings()
    return _settings


def today():
    """Get today's date - wrapper to make testing easier."""
    return date.today()


def extract_report_date(content: str) -> Optional[date]:
    """Extract and validate report date from report content.

    Args:
        content (str): Report content to parse

    Returns:
        date | None: Extracted report date, or None if date pattern not found

    Raises:
        ValueError: If date components are invalid (e.g., month > 12)
    """
    match = REPORT_DATE_PATTERN.search(content)

    if not match:
        get_logger(LOGGER_NAME).warning("Could not find report date in content")
        return None

    month, day, year = match.groups()
    try:
        full_year = 2000 + int(year)  # Convert 2-digit year to 4-digit year (assuming 20xx)
        month_int = int(month)
        day_int = int(day)

        if month_int < 1 or month_int > 12:
            raise ValueError("month must be in 1..12")

        # Get max days for this month
        max_days = 31
        if month_int in [4, 6, 9, 11]:
            max_days = 30
        elif month_int == 2:
            # Check for leap year
            max_days = 29 if (full_year % 4 == 0 and full_year % 100 != 0) or (full_year % 400 == 0) else 28

        if day_int < 1 or day_int > max_days:
            if month_int == 2 and day_int == 29:
                raise ValueError("day is out of range for month")
            raise ValueError("day must be in 1..31")

        return date(full_year, month_int, day_int)
    except ValueError as e:
        get_logger(LOGGER_NAME).warning(f"Invalid date components: {str(e)}")
        raise


def retry_operation(interval: int, operation: Callable[[], T], max_attempts: int = 0) -> T:
    """Execute an operation with retry logic using the shared utility.

    Args:
        interval (int): Seconds to wait between attempts
        operation (Callable[[], T]): Function to execute
        max_attempts (int, optional): Maximum number of attempts, 0 for unlimited. Defaults to 0.

    Returns:
        T: Result of the operation

    Raises:
        ValueError: If max_attempts is negative
        Exception: Last error encountered if max attempts reached
    """
    return with_retry(interval, operation, max_attempts, LOGGER_NAME)


def ftp_connect(server: Server, operation: Callable[[FTP], T], timeout: int = NETWORK_TIMEOUT) -> T:
    """Execute an operation with an FTP connection using the shared utility.

    Args:
        server (Server): Server connection details (must use FTP protocol)
        operation (Callable[[FTP], T]): Lambda to execute with FTP connection
        timeout (int): Timeout for network operations

    Returns:
        T: Result of the operation lambda

    Raises:
        ValueError: If server protocol is not FTP or if server attributes are invalid
        ftplib.all_errors: Any FTP-related errors that occur during connection or operation
    """
    return with_ftp_connection(server, operation, timeout, LOGGER_NAME)


def sftp_connect(server: Server, operation: Callable[[SFTPClient], T], timeout: int = NETWORK_TIMEOUT) -> T:
    """Execute an operation with an SFTP connection using the shared utility.

    Args:
        server (Server): Server connection details (must use SFTP protocol)
        operation (Callable[[SFTPClient], T]): Lambda to execute with SFTP connection
        timeout (int): Timeout for network operations

    Returns:
        T: Result of the operation lambda

    Raises:
        ValueError: If server protocol is not SFTP or if server attributes are invalid
        paramiko.SSHException: SSH/SFTP connection errors
        socket.error: Network-related errors
    """
    return with_sftp_connection(server, operation, timeout, LOGGER_NAME)


def write_string_to_path(path: Path, content: str):
    """Write string content to a file path using the shared utility.

    Args:
        path (Path): Path to write content to
        content (str): Content to write
    """
    write_str_to_path(path, content, LOGGER_NAME)


def gen_rpt_filename(report_id: str, report_date: date, _store_number: str) -> str:
    """Generate standardized report filename.

    Args:
        report_id (str): Report identifier
        report_date (date): Report date
        _store_number (str): Store number (must be at least 3 digits)

    Returns:
        str: Generated filename in format [id][YYMMDD].[last 3 digits of store]

    Raises: n
        IndexError: If store number has fewer than 3 digits
    """
    if len(_store_number) < 3:
        raise IndexError("Store number must have at least 3 digits")

    store_suffix = _store_number[-3:]
    return f"{report_id}{report_date.strftime('%y%m%d')}.{store_suffix}.html"


def html_wrap_content(content: str) -> str:
    """Wrap content for inclusion in HTML.

    Args:
        content (str): Content to wrap

    Returns:
        str: Wrapped content with HTML characters escaped
    """
    # Escape HTML characters
    escaped_content = content.replace("<", "&lt;").replace(">", "&gt;")
    return f"<pre>{escaped_content}</pre>"


def filter_tlog_content(_content: str) -> str:
    """Filter out unwanted content from raw tlog content.

    Args:
        _content (str): Raw transaction log content

    Returns:
        str: Filtered transaction log content
    """
    return '\n'.join(line for line in _content.splitlines() if not any(p.match(line) for p in TLOG_FILTER_PATTERNS))


def extract_transactions(_content: str) -> List[Transaction]:
    """Extract, index, and format transactions from raw tlog content.

    Args:
        _content (str): Raw transaction log content, filtered by filter_tlog_content

    Returns:
        List[Transaction]: List of transactions, each containing:
            - index (int): Transaction index, 1-based
            - detail (str): Transaction detail
    """

    details = [b for b in filter_tlog_content(_content).split(TLOG_TX_SEPARATOR) if b.strip()]

    return [
        Transaction(index, detail)
        for index, detail in enumerate(details, start=1)
        if len(detail.splitlines()) >= 3
    ]


def generate_indexed_tlog(transactions: List[Transaction]) -> str:
    """Generate indexed transaction log.

    Args:
        transactions (List[Transaction]): List of transactions to format

    Returns:
        str: Indexed transaction log content
    """
    return html_wrap_content('\n'.join(t.detail_html() for t in transactions))


def generate_vrdis(_transactions: List[Transaction], rel_tlog_path: str,
                   inclusion_patterns: List[Pattern[str]]) -> str:
    """Generate VRDIS report with links to full transaction log.

    Args:
        _transactions (List[Transaction]): List of transactions to format
        rel_tlog_path (str): Relative path to transaction log
        inclusion_patterns (List[Pattern[str]]): List of inclusion patterns; if transaction detail matches any of
        these patterns, it will be included in the report

    Returns:
        str: VRDIS report content
    """
    vrdis_transactions = []
    for tx in _transactions:
        if any(p.search(tx.detail) for p in inclusion_patterns):
            vrdis_entry = (
                f"{tx.detail_html()}"
                f'<a href="{rel_tlog_path}#{tx.index}">'
                f'<span class="tlog-link">View this transaction in full transaction log</span></a>'
                f'<div style="height: 15px"></div>'
            )
            vrdis_transactions.append(vrdis_entry)

    return html_wrap_content('\n'.join(vrdis_transactions))


def ftp_file_exists(ftp: ftplib.FTP, path: str) -> bool:
    """Check if a file exists on FTP server without raising an exception.

    Args:
        ftp (ftplib.FTP): Active FTP connection
        path (str): Remote file path

    Returns:
        bool: True if file exists, False otherwise
    """
    logger = get_logger(LOGGER_NAME)

    try:
        logger.debug(f"Checking for {path} on FTP server")
        ftp.size(path)  # or ftp.nlst(path)
        logger.debug(f"Found {path} on FTP server")
        return True
    except ftplib.error_perm:
        logger.debug(f"Could not find {path} on FTP server")
        return False


def ftp_download_file(ftp: FTP, path: str) -> str:
    """Download a file from the FTP server.

    Args:
        ftp (FTP): Active FTP connection
        path (str): Path to the file on the FTP server

    Returns:
        str: Content of the file
    """
    logger = get_logger(LOGGER_NAME)

    logger.info(f"Downloading {path}")
    content = io.BytesIO()
    ftp.retrbinary(f"RETR {path}", content.write)
    content_str = content.getvalue().decode('cp1252')
    content.close()
    logger.info(f"Downloaded {path}")
    return content_str


def ftp_get_todays_report(ftp: FTP, path: str) -> str:
    """Get today's report from the FTP server. Raise FileNotFoundError if file doesn't exist or isn't for today's date.

    Args:
        ftp (FTP): Active FTP connection
        path (str): Path to the report on the FTP server

    Returns:
        str: Content of the report
    """
    if ftp_file_exists(ftp, path):
        content = ftp_download_file(ftp, path)
        report_date = extract_report_date(content)
        if not report_date:
            raise FileNotFoundError(f"Report date not found in {path}")
        elif report_date != today():
            raise FileNotFoundError(f"Report date {report_date} does not match today's date for {path}")

        return content
    else:
        raise FileNotFoundError(f"Report not found at {path}")


def sftp_upload_file(sftp: SFTPClient, remote_path: str, content: str) -> None:
    """Upload a file to the SFTP server.

    Args:
        sftp (SFTPClient): SFTP client
        remote_path (str): Path to the file on the SFTP server
        content (str): Content to write to the file
    """
    logger = get_logger(LOGGER_NAME)

    logger.info(f"Uploading {len(content)} bytes to {remote_path}")
    with sftp.file(remote_path, 'w') as f:
        f.write(content)
    logger.info(f"Successfully uploaded {len(content)} bytes to {remote_path}")


def process_report(
        content: str,
        file_spec: FileSpec,
        report_date: date,
        _store_number: str,
        _upload_queue: List[UploadQueueItem],
        local_reports_root: str,
        tlog_id: str,
        vrdis_id: str,
        vrdis_name: str,
        vrdis_inclusion_patterns: List[Pattern[str]]
) -> List[Tuple[Path, str]]:
    """Process a single report file.

    Args:
        content (str): Report content
        file_spec (FileSpec): Report specification
        report_date (date): Report date
        _store_number (str): Store number (for report filename generation)
        _upload_queue (List[UploadQueueItem]): Queue for files to be uploaded
        local_reports_root (str): Path to the local reports root
        tlog_id (str): Transaction log report ID
        vrdis_id (str): VRDIS report ID
        vrdis_name (str): VRDIS report name
        vrdis_inclusion_patterns (List[Pattern[str]]): List of inclusion patterns; if transaction detail matches any
        of these patterns, it will be included in the report

    Returns:
        List[Tuple[Path, str]]: List of (path, content) tuples to be written

    Raises:
        ValueError: If report date doesn't match today's date
    """
    logger = get_logger(LOGGER_NAME)

    logger.info(f"Processing report {file_spec.name}")
    writes: List[Tuple[Path, str]] = []

    try:
        report_filename = gen_rpt_filename(file_spec.id, report_date, _store_number)

        if file_spec.id == tlog_id:
            # t-log processing
            transactions = extract_transactions(content)
            indexed_tlog_content = generate_indexed_tlog(transactions)
            writes.extend([
                (Path(local_reports_root) / report_filename, indexed_tlog_content),
                (Path(local_reports_root) / file_spec.name / report_filename, indexed_tlog_content)
            ])

            # VRDIS report generation
            vrdis_filename = gen_rpt_filename(vrdis_id, report_date, _store_number)
            writes.extend([
                (
                    Path(local_reports_root) / vrdis_filename,
                    generate_vrdis(transactions, f"{file_spec.name}/{report_filename}", vrdis_inclusion_patterns)
                ),
                (
                    Path(local_reports_root) / vrdis_name / vrdis_filename,
                    generate_vrdis(transactions, f"../{file_spec.name}/{report_filename}", vrdis_inclusion_patterns)
                )
            ])
        else:
            # regular report processing
            wrapped_content = html_wrap_content(content)
            writes.extend([
                (Path(local_reports_root) / report_filename, wrapped_content),
                (Path(local_reports_root) / file_spec.name / report_filename, wrapped_content)
            ])

        if file_spec.upload:
            _upload_queue.append(UploadQueueItem(content, report_filename))
            logger.debug(f"Added {report_filename} to upload queue")

        return writes
    except Exception as e:
        logger.error(f"Error processing {file_spec.name}: {str(e)}")
        raise


def process_reports(
        ftp: FTP,
        _files: List[FileSpec],
        process_file: Callable[[str, FileSpec, date], List[Tuple[Path, str]]]
) -> None:
    """Download and process reports from controller via FTP.

    Args:
        ftp (FTP): Active FTP connection
        _files (List[FileSpec]): List of files to process
        process_file (Callable[[str, FileSpec], List[Tuple[Path, str]]]): Function to process each downloaded file

    Raises:
        IOError: If file operations fail
    """
    logger = get_logger(LOGGER_NAME)

    logger.info(f"Processing {len(_files)} reports")

    for file_spec in _files:
        try:
            content = ftp_get_todays_report(ftp, file_spec.path)

            # Process file and get writes to perform
            writes = process_file(content, file_spec, today())

            # Perform all writes for this report
            for path, content in writes:
                logger.debug(f"Writing to {path}")
                path.parent.mkdir(parents=True, exist_ok=True)
                path.write_text(content)
        except FileNotFoundError:
            if file_spec.required:
                logger.error(f"Required file {file_spec.name} not found at {file_spec.path}")
                raise
            else:
                logger.warning(f"Optional file {file_spec.name} not found at {file_spec.path}, ignoring")
        except (TimeoutError, ConnectionError) as e:
            logger.error(f"Network error downloading {file_spec.name}: {str(e)}")
            raise
        except ftplib.all_errors as e:
            logger.error(f"FTP error downloading {file_spec.name}: {str(e)}")
            raise
        except Exception as e:
            logger.error(f"Error downloading or processing {file_spec.name}: {str(e)}")
            raise


def process_upload_queue(server: Server, _upload_queue: List[UploadQueueItem], timeout: int = 30) -> None:
    """Upload reports to remote SFTP server.

    Args:
        server (Server): Remote server connection details
        _upload_queue (List[Dict[str, str]]): List of files to upload, each containing 'content' and 'filename'
        timeout (int): Timeout for network operations

    Raises:
        ValueError: If server protocol is not SFTP
        SSHException: If SFTP connection or transfer fails
    """
    logger = get_logger(LOGGER_NAME)

    if not _upload_queue:
        logger.info("Upload queue is empty, skipping")
        return

    logger.info(f"Processing upload queue containing {len(_upload_queue)} items")

    def upload_files(sftp: SFTPClient) -> None:
        for item in _upload_queue:
            sftp_upload_file(sftp, f"{server.path}/{item.filename}", item.content)

    with_sftp_connection(server, upload_files, timeout)


def pull_and_process_reports(upload_queue: List[UploadQueueItem]) -> None:
    """Pull reports from controller, process them.

    Args:
        upload_queue (List[UploadQueueItem]): Queue for files to be uploaded
    """
    settings = get_settings()
    vrdis_inclusion_patterns = [re.compile(p) for p in settings.vrdis_inclusion_patterns]

    logger = get_logger(LOGGER_NAME)
    logger.info(f"Downloading and processing reports from {settings.cc.host}")
    with_ftp_connection(
        settings.cc,
        lambda ftp: process_reports(
            ftp,
            settings.reports,
            lambda content, file_spec, report_date: process_report(
                content,
                file_spec,
                report_date,
                settings.store_number,
                upload_queue,
                settings.local_reports_root,
                settings.tlog_id,
                settings.vrdis_id,
                settings.vrdis_name,
                vrdis_inclusion_patterns
            )
        ),
        settings.network_timeout
    )


def delete_local_reports(folder: Path) -> None:
    """Delete existing reports (*.html) from specified folder.

    Args:
        folder (Path): Path to the directory containing reports
    """
    logger = get_logger(LOGGER_NAME)

    if not folder.exists():
        return

    for report in folder.glob("*.html"):
        try:
            logger.info(f"Deleting  report {report}")
            report.unlink()
        except PermissionError:
            logger.error(f"Permission denied deleting {report}")
        except Exception as e:
            logger.error(f"Error deleting {report}: {str(e)}")


def main():
    logger = get_logger(LOGGER_NAME)
    settings = get_settings()

    try:
        logger.info(f"Starting ACE reports processing for store {settings.store_number}")

        # delete existing reports from local reports root
        logger.info(f"Deleting existing reports from local reports root folder {settings.local_reports_root}")
        delete_local_reports(Path(settings.local_reports_root))

        upload_queue = []

        # pull reports from controller, process them
        with_retry(
            settings.retry_interval,
            lambda: pull_and_process_reports(upload_queue)
        )

        # process the upload queue
        with_retry(
            settings.retry_interval,
            lambda: process_upload_queue(
                settings.reports_server,
                upload_queue,
                settings.network_timeout
            )
        )

        logger.info("Processing of ACE reports completed successfully")
    except Exception as e:
        logger.error(f"Failed to process ACE reports: {str(e)}")
        raise


if __name__ == "__main__":
    settings = get_settings()
    log_path = Path(settings.log_directory) / settings.log_base_filename
    setup_logging(str(log_path), LOGGER_NAME, settings.log_count)
    try:
        main()
    except Exception as e:
        get_application_logger().error(f"Failed to process reports: {str(e)}")
        raise
