"""
Common utilities for CommServ store utilities.

This module provides shared functionality used across multiple scripts in the CommServ store utilities package.
"""
import ftplib
import io
import logging
import time
from dataclasses import dataclass
from logging.handlers import TimedRotatingFileHandler
from pathlib import Path
from typing import TypeVar, Callable, Optional, List

import paramiko
from paramiko import SSHClient, AutoAddPolicy
from paramiko.sftp_client import SFTPClient
from pyhocon import ConfigFactory, ConfigException

T = TypeVar('T')


@dataclass
class Server:
    """Represents connection details for a remote server.

    Attributes:
        protocol (str): Connection protocol (e.g. 'ftp', 'sftp')
        host (str): Server hostname or IP address
        username (str): Authentication username
        password (str): Authentication password
        port (int): Server port
        path (str): Base path on server
    """
    protocol: str
    host: str
    username: str
    password: str
    port: int
    path: str
    
    def validate(self, required_protocol: Optional[str] = None):
        """Validate server configuration.

        Args:
            required_protocol: (Optional) Protocol that this server must use ('ftp' or 'sftp'). Defaults to None.

        Raises:
            ValueError: If any configuration values are invalid
        """
        if required_protocol and self.protocol != required_protocol:
            raise ValueError(f"Server must use {required_protocol.upper()} protocol")
        if not self.host:
            raise ValueError("Host cannot be empty")
        if not self.username:
            raise ValueError("Username cannot be empty")
        if not self.password:
            raise ValueError("Password cannot be empty")
        if self.port <= 0 or self.port > 65535:
            raise ValueError("Port must be between 1 and 65535")


@dataclass
class StoreSettings:
    """Shared settings for all CommServ store utilities.

    Attributes:
        store_number (str): Store number
        cc (Server): CC controller FTP configuration
        network_timeout (int): Timeout in seconds for network operations
        retry_interval (int): Interval in seconds between retry attempts
        max_attempts (int): Maximum number of retry attempts (0 for unlimited)
        log_directory (str): Directory for log files
        log_count (int): Number of backup log files to keep
    """
    store_number: str
    cc: Server
    network_timeout: int
    retry_interval: int
    max_attempts: int
    log_directory: str
    log_count: int


def setup_logging(log_file: str, logger_name: str = "commserv-utils", log_count: int = 5):
    """Configure logging for a script.

    Args:
        log_file (str): Path to the log file
        logger_name (str): Name of the logger
        log_count (int): Number of backup log files to keep
    """
    # Ensure log directory exists
    Path(log_file).parent.mkdir(parents=True, exist_ok=True)

    # Create logger
    logger = logging.getLogger(logger_name)
    logger.setLevel(logging.DEBUG)

    # Remove existing handlers if any
    for handler in logger.handlers[:]:
        logger.removeHandler(handler)

    # Create formatters and add it to the handler
    formatter = logging.Formatter('%(asctime)s %(name)-12s %(levelname)-8s %(message)s')
    console_handler = logging.StreamHandler()
    console_handler.setFormatter(formatter)

    # File handler with daily rotation
    file_handler = TimedRotatingFileHandler(
        log_file,
        when="midnight",
        interval=1,
        backupCount=log_count
    )
    file_handler.setFormatter(formatter)

    # Add both handlers to the logger
    logger.addHandler(console_handler)
    logger.addHandler(file_handler)

    # Log delimiter for new execution
    logger.info("-" * 80)
    logger.info("New execution started")


def get_logger(logger_name: str = "commserv-utils"):
    """Get the application logger.
    
    Args:
        logger_name (str): Name of the logger to retrieve
        
    Returns:
        logging.Logger: The logger instance
    """
    return logging.getLogger(logger_name)


def with_retry(
        interval: int,
        operation: Callable[[], T],
        max_attempts: int = 0,
        logger_name: str = "commserv-utils"
) -> T:
    """Execute an operation with retry logic.

    Args:
        interval (int): Seconds to wait between attempts
        operation (Callable[[], T]): Function to execute
        max_attempts (int, optional): Maximum number of attempts, 0 for unlimited. Defaults to 0.
        logger_name (str): Name of the logger to use

    Returns:
        T: Result of the operation

    Raises:
        ValueError: If max_attempts is negative
        Exception: Last error encountered if max attempts reached
    """
    logger = get_logger(logger_name)

    if max_attempts < 0:
        raise ValueError("max_attempts cannot be negative")

    attempt = 1

    while True:
        try:
            if max_attempts > 0:
                debug_msg = f"Attempt {attempt} of {max_attempts}"
            else:
                debug_msg = f"Attempt {attempt}"
            logger.debug(debug_msg)
            del debug_msg

            return operation()

        except Exception as e:
            # Get error details and immediately clear the exception
            error_msg = str(e)
            error_cls = e.__class__
            del e

            logger.warning(f"Attempt {attempt} failed: {error_msg}")

            if 0 < max_attempts <= attempt:
                logger.error(f"Maximum attempts ({max_attempts}) reached")
                raise error_cls(error_msg)

            logger.info(f"Retrying in {interval} seconds...")
            time.sleep(interval)
            attempt += 1

            # Clear references
            del error_msg
            del error_cls


def with_ftp_connection(
        server: Server, 
        operation: Callable[[ftplib.FTP], T], 
        timeout: int = 30,
        logger_name: str = "commserv-utils"
) -> T:
    """Execute an operation with an FTP connection that is automatically closed.

    Args:
        server (Server): Server connection details (must use FTP protocol)
        operation (Callable[[FTP], T]): Lambda to execute with FTP connection
        timeout (int): Timeout for network operations
        logger_name (str): Name of the logger to use

    Returns:
        T: Result of the operation lambda

    Raises:
        ValueError: If server protocol is not FTP or if server attributes are invalid
        ftplib.all_errors: Any FTP-related errors that occur during connection or operation
    """
    logger = get_logger(logger_name)

    server.validate("ftp")

    logger.info(f"Connecting to FTP server {server.host}")
    ftp = ftplib.FTP()
    connected = False
    try:
        ftp.connect(server.host, server.port, timeout)
        connected = True
        ftp.login(server.username, server.password)
        logger.debug("FTP connection established")
        return operation(ftp)
    except ftplib.all_errors as e:
        logger.error(f"FTP error: {str(e)}")
        raise
    finally:
        if connected:
            try:
                logger.debug("Closing FTP connection")
                ftp.quit()
            except:
                # Ignore errors during quit
                pass


def with_sftp_connection(
        server: Server, 
        operation: Callable[[SFTPClient], T], 
        timeout: int = 30,
        logger_name: str = "commserv-utils"
) -> T:
    """Execute an operation with an SFTP connection that is automatically closed.

    Args:
        server (Server): Server connection details (must use SFTP protocol)
        operation (Callable[[SFTPClient], T]): Lambda to execute with SFTP connection
        timeout (int): Timeout for network operations
        logger_name (str): Name of the logger to use

    Returns:
        T: Result of the operation lambda

    Raises:
        ValueError: If server protocol is not SFTP or if server attributes are invalid
        paramiko.SSHException: SSH/SFTP connection errors
        socket.error: Network-related errors
    """
    logger = get_logger(logger_name)

    server.validate("sftp")

    logger.info(f"Connecting to SFTP server {server.host}")
    ssh = SSHClient()
    ssh.set_missing_host_key_policy(AutoAddPolicy())
    connected = False

    try:
        ssh.connect(
            hostname=server.host,
            port=server.port,
            username=server.username,
            password=server.password,
            timeout=timeout,
            allow_agent=False,
            look_for_keys=False
        )
        connected = True
        sftp = ssh.open_sftp()
        logger.debug("SFTP connection established")
        return operation(sftp)
    except Exception as e:
        logger.error(f"SFTP error: {str(e)}")
        raise
    finally:
        if connected:
            try:
                logger.debug("Closing SFTP connection")
                ssh.close()
            except:
                # Ignore errors during close
                pass


def ftp_download_file(ftp: ftplib.FTP, path: str, logger_name: str = "commserv-utils") -> str:
    """Download a file from the FTP server.

    Args:
        ftp (FTP): Active FTP connection
        path (str): Path to the file on the FTP server
        logger_name (str): Name of the logger to use

    Returns:
        str: Content of the file
    """
    logger = get_logger(logger_name)
    logger.info(f"Downloading file: {path}")
    
    buffer = io.BytesIO()
    ftp.retrbinary(f"RETR {path}", buffer.write)
    buffer.seek(0)
    content = buffer.read().decode('utf-8')
    
    logger.debug(f"Downloaded {len(content)} bytes")
    return content


def ftp_upload_file(ftp: ftplib.FTP, path: str, content: str, logger_name: str = "commserv-utils"):
    """Upload a file to the FTP server.

    Args:
        ftp (FTP): Active FTP connection
        path (str): Path to the file on the FTP server
        content (str): Content to upload
        logger_name (str): Name of the logger to use
    """
    logger = get_logger(logger_name)
    logger.info(f"Uploading to path: {path}")
    
    buffer = io.BytesIO(content.encode())
    ftp.storbinary(f"STOR {path}", buffer)
    
    logger.debug(f"Uploaded {len(content)} bytes")


def sftp_download_file(sftp: SFTPClient, path: str, logger_name: str = "commserv-utils") -> bytes:
    """Download a file from the SFTP server.

    Args:
        sftp (SFTPClient): Active SFTP connection
        path (str): Path to the file on the SFTP server
        logger_name (str): Name of the logger to use

    Returns:
        bytes: Content of the file
    """
    logger = get_logger(logger_name)
    logger.info(f"Downloading file: {path}")
    
    buffer = io.BytesIO()
    sftp.getfo(path, buffer)
    buffer.seek(0)
    content = buffer.read()
    
    logger.debug(f"Downloaded {len(content)} bytes")
    return content


def sftp_upload_file(sftp: SFTPClient, path: str, content: bytes, logger_name: str = "commserv-utils"):
    """Upload a file to the SFTP server.

    Args:
        sftp (SFTPClient): Active SFTP connection
        path (str): Path to the file on the SFTP server
        content (bytes): Content to upload
        logger_name (str): Name of the logger to use
    """
    logger = get_logger(logger_name)
    logger.info(f"Uploading to path: {path}")
    
    buffer = io.BytesIO(content)
    sftp.putfo(buffer, path)
    
    logger.debug(f"Uploaded {len(content)} bytes")


def write_str_to_path(path: Path, content: str, logger_name: str = "commserv-utils"):
    """Write string content to a file path, creating parent directories if necessary.

    Args:
        path (Path): Path to write content to
        content (str): Content to write
        logger_name (str): Name of the logger to use
    """
    logger = get_logger(logger_name)
    try:
        path.parent.mkdir(parents=True, exist_ok=True)
        with open(path, 'w', encoding='utf-8') as f:
            f.write(content)
        logger.info(f"Wrote {len(content)} bytes to {path}")
    except Exception as e:
        logger.error(f"Error writing to {path}: {str(e)}")
        raise


def load_store_settings() -> StoreSettings:
    """Load shared store settings from the store.conf file.

    Returns:
        StoreSettings: Shared store settings

    Raises:
        FileNotFoundError: If config file doesn't exist
        ConfigException: If config file has invalid format
    """
    config_path = Path(__file__).parent.parent / "etc" / "store.conf"
    config = ConfigFactory.parse_file(str(config_path))
    
    return StoreSettings(
        store_number=config.get_string("store_number"),
        cc=Server(
            protocol="ftp",
            host=config.get_string("cc.host"),
            port=config.get_int("cc.port", 21),
            username=config.get_string("cc.username"),
            password=config.get_string("cc.password"),
            path="/"
        ),
        network_timeout=config.get_int("network_timeout"),
        retry_interval=config.get_int("retry_interval"),
        max_attempts=config.get_int("max_attempts"),
        log_directory=config.get_string("log_directory"),
        log_count=config.get_int("log_count")
    )


# Store settings at module level
_store_settings = None


def get_store_settings() -> StoreSettings:
    """Get shared store settings, loading them if necessary.

    Returns:
        StoreSettings: Shared store settings
    """
    global _store_settings
    if _store_settings is None:
        _store_settings = load_store_settings()
    return _store_settings
