from dataclasses import dataclass
from pathlib import Path
import io
import zipfile
import sys
from pyhocon import ConfigFactory, ConfigException

# Import shared utilities
try:
    # Try relative import first (when used as a package)
    from .commserv_utils import (
        Server, StoreSettings, setup_logging, get_logger, with_retry,
        with_ftp_connection, with_sftp_connection, get_store_settings
    )
except ImportError:
    # Fall back to absolute import (when run as a script)
    from commserv_utils import (
        Server, StoreSettings, setup_logging, get_logger, with_retry,
        with_ftp_connection, with_sftp_connection, get_store_settings
    )


@dataclass
class Settings(StoreSettings):
    """Settings for the update_wic_apl script.

    Inherits all shared settings from StoreSettings and adds application-specific settings.

    Attributes:
        apl_server (Server): SFTP server connection details for APL files
        agency_id (str): Agency identifier
        log_base_filename (str): Base filename for logs
    """
    apl_server: Server
    agency_id: str
    log_base_filename: str


def load_settings() -> Settings:
    """Load configuration from HOCON files.

    Loads shared settings from store.conf and application-specific settings from update_wic_apl.conf.

    Returns:
        Settings: Combined configuration

    Raises:
        FileNotFoundError: If any config file doesn't exist
        ConfigException: If any config file has invalid format
    """
    # Get shared store settings
    store_settings = get_store_settings()
    
    # Load app-specific settings
    # Try multiple possible locations for the config file
    possible_paths = [
        Path(__file__).parent.parent / "etc" / "update_wic_apl.conf",  # Standard relative path
        Path("etc") / "update_wic_apl.conf",                          # Relative to current directory
        Path("C:/Commserv/etc/update_wic_apl.conf"),                  # Absolute path for Windows
    ]
    
    config = None
    errors = []
    
    for path in possible_paths:
        try:
            if path.exists():
                print(f"Loading configuration from {path}")
                config = ConfigFactory.parse_file(str(path))
                break
        except Exception as e:
            errors.append(f"Error loading {path}: {str(e)}")
    
    if config is None:
        error_msg = "\n".join(["Could not find or parse update_wic_apl.conf in any of these locations:"] + 
                            [f"- {p} {'(exists)' if p.exists() else '(not found)'}" for p in possible_paths] + 
                            ["\nErrors encountered:"] + errors)
        raise FileNotFoundError(error_msg)
    
    # Create combined settings
    return Settings(
        # Shared settings from store.conf
        store_number=store_settings.store_number,
        cc=store_settings.cc,
        network_timeout=store_settings.network_timeout,
        retry_interval=store_settings.retry_interval,
        max_attempts=store_settings.max_attempts,
        log_directory=store_settings.log_directory,
        log_count=store_settings.log_count,
        
        # App-specific settings from update_wic_apl.conf
        apl_server=Server(
            protocol="sftp",
            host=config.get_string("apl_server.host"),
            port=config.get_int("apl_server.port"),
            username=config.get_string("apl_server.username"),
            password=config.get_string("apl_server.password"),
            path="/"
        ),
        agency_id=config.get_string("agency_id"),
        log_base_filename=config.get_string("log_base_filename")
    )


# Load settings at module level
_settings = None


def get_settings() -> Settings:
    """Get application settings, loading them if necessary."""
    global _settings
    if _settings is None:
        _settings = load_settings()
    return _settings


# Constants
LOGGER_NAME = "mu-wic-apl"


def find_and_download_apl_zip():
    """Find and download the APL ZIP file from the SFTP server.
    
    Returns:
        tuple: A tuple containing (zip_content, apl_filename)
    """
    logger = get_logger(LOGGER_NAME)
    settings = get_settings()
    
    def sftp_operation(sftp_client):
        # List directory and find matching file
        files = sftp_client.listdir()
        zip_files = [f for f in files if f.endswith('_APL_04b.zip')]

        if not zip_files:
            raise Exception("No matching ZIP files found")

        target_zip = zip_files[0]
        logger.info(f"Found ZIP file: {target_zip}")

        # Download ZIP in memory
        memory_file = io.BytesIO()
        sftp_client.getfo(target_zip, memory_file)
        memory_file.seek(0)
        
        # Extract ZIP contents
        with zipfile.ZipFile(memory_file) as zip_ref:
            if len(zip_ref.namelist()) != 1:
                raise Exception("ZIP file must contain exactly one file")

            apl_filename = zip_ref.namelist()[0]
            logger.info(f"Extracted file: {apl_filename}")

            with zip_ref.open(apl_filename) as apl_file:
                apl_content = apl_file.read().decode('utf-8')
                
        return apl_content, apl_filename
    
    return with_retry(
        interval=settings.retry_interval,
        operation=lambda: with_sftp_connection(
            server=settings.apl_server,
            operation=sftp_operation,
            timeout=settings.network_timeout,
            logger_name=LOGGER_NAME
        ),
        max_attempts=settings.max_attempts,
        logger_name=LOGGER_NAME
    )


def upload_apl_to_cc(apl_content, apl_filename):
    """Upload the APL file to the CC server.
    """
    logger = get_logger(LOGGER_NAME)
    settings = get_settings()
    
    def ftp_operation(ftp):
        # Prepare upload path
        upload_path = f"c:/adx_idt1/{settings.agency_id}{apl_filename[:4]}.apl"
        logger.info(f"Uploading to path: {upload_path}")

        # Upload content
        ftp.storbinary(f'STOR {upload_path}', io.BytesIO(apl_content.encode()))

        # Send command
        logger.info("Sending ADXSTART command")
        ftp.sendcmd("ADXSTART acewerbl")
        
        return True
    
    return with_retry(
        interval=settings.retry_interval,
        operation=lambda: with_ftp_connection(
            server=settings.cc,
            operation=ftp_operation,
            timeout=settings.network_timeout,
            logger_name=LOGGER_NAME
        ),
        max_attempts=settings.max_attempts,
        logger_name=LOGGER_NAME
    )


def main():
    """Main function to update MU WIC APL files."""
    logger = get_logger(LOGGER_NAME)
    
    try:
        logger.info("Script started")
        
        # Download the APL file from SFTP
        apl_content, apl_filename = find_and_download_apl_zip()
        
        # Upload the APL file to CC
        upload_apl_to_cc(apl_content, apl_filename)
        
        logger.info("Script completed successfully")
        return 0
        
    except Exception as e:
        logger.error(f"Error occurred: {str(e)}", exc_info=True)
        return 1


if __name__ == "__main__":
    settings = get_settings()
    log_path = Path(settings.log_directory) / settings.log_base_filename
    setup_logging(str(log_path), LOGGER_NAME, settings.log_count)
    try:
        exit_code = main()
        sys.exit(exit_code)
    except Exception as e:
        get_logger(LOGGER_NAME).error(f"Failed to update APL: {str(e)}")
        sys.exit(1)